//----------------------------------------------------------
// Win32 Service Class
// Written by Christopher Kohlhoff
//
// Implementation of the TService class, a class that
// encapsulates much of the code needed for writing Win32
// services.
//----------------------------------------------------------

#include <owl/pch.h>
#include <winsys/registry.h>
#include <stdio.h>
#include "service.h"


//
// The pointer to the one and only service instance
//

static TService* Service;


//
// Service callback functions
//

void WINAPI
ServiceMain(uint32 argc, char* argv[])
{
  Service->ServiceMain(argc, argv);
}


void WINAPI
ControlHandler(uint32 control)
{
  Service->ControlHandler(control);
}


//
// Class to manage service handles
//

class TServiceHandle {
  public:
    TServiceHandle(SC_HANDLE handle) : Handle(handle) {}
    ~TServiceHandle() { if (Handle) CloseServiceHandle(Handle); }
    operator SC_HANDLE() { return Handle; }

  private:
    SC_HANDLE Handle;
};


//
// TService class implementation
//

TService::TService()
:
  StopFlag(true),
  HStatus(0),
  StatusCode(SERVICE_STOPPED),
  ErrorCode(0),
  HEventLog(0),
  ArgC(0),
  ArgV(0),
  PauseEvent(true)
{
  ::Service = this;
}


TService::~TService()
{
  if (HEventLog)
    DeregisterEventSource(HEventLog);
}


bool
TService::Start()
{
  // set the service entry point and start it
  //
  SERVICE_TABLE_ENTRY entries[2];
  entries[0].lpServiceName = (char*) GetName();
  entries[0].lpServiceProc = ::ServiceMain;
  entries[1].lpServiceName = NULL;
  entries[1].lpServiceProc = NULL;
  return StartServiceCtrlDispatcher(entries);
}


bool
TService::Start(int argc, char* argv[])
{
  // run the service on the console
  //
  PauseEvent.Set();
  StopFlag = false;
  ArgC = argc;
  ArgV = argv;
  if (!Init())
    return false;
  Run();
  Cleanup();
  return true;
}


bool
TService::Register()
{
  TServiceHandle scm(OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS));
  if (scm == NULL)
    return false;

  if (!CreateService(scm) && GetLastError() != ERROR_SERVICE_EXISTS)
    return false;

  // register for the event log
  //
  char path[MAX_PATH];
  char buf[MAX_PATH];
  GetModuleFileName(NULL, path, MAX_PATH);
  uint flags = EVENTLOG_ERROR_TYPE |
               EVENTLOG_WARNING_TYPE |
               EVENTLOG_INFORMATION_TYPE;
  sprintf(buf, "SYSTEM\\CurrentControlSet\\Services\\EventLog\\Application\\%s",
          GetName());
  TRegKey key(HKEY_LOCAL_MACHINE, buf);
  key.SetValue("TypesSupported", REG_DWORD, (uint8*)&flags, sizeof(DWORD));
  key.SetValue("EventMessageFile", REG_SZ, (uint8*)path, strlen(path) + 1);

  return true;
}


bool
TService::Unregister()
{
  TServiceHandle scm(OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS));
  if (scm == NULL)
    return false;

  // open and delete the service
  //
  TServiceHandle svc(OpenService(scm, GetName(), SERVICE_ALL_ACCESS));
  if (GetLastError() != ERROR_SERVICE_DOES_NOT_EXIST) {
    if (svc == NULL)
      return false;
    DeleteService(svc);
  }

  // deregister for the event log
  //
  TRegKey key(HKEY_LOCAL_MACHINE,
              "SYSTEM\\CurrentControlSet\\Services\\EventLog\\Application");
  key.DeleteKey(GetName());

  return true;
}


bool
TService::CreateService(SC_HANDLE scm)
{
  char imagePath[MAX_PATH];
  GetModuleFileName(NULL, imagePath, MAX_PATH);
  return ::CreateService(scm, GetName(), GetDisplayName(),
                         SERVICE_ALL_ACCESS, SERVICE_WIN32_OWN_PROCESS,
                         SERVICE_DEMAND_START, SERVICE_ERROR_NORMAL,
                         imagePath, NULL, NULL, NULL, NULL, NULL);
}


bool
TService::Init()
{
  return true;
}


void
TService::Cleanup()
{
}


void
TService::PauseBegin()
{
}


void
TService::PauseEnd()
{
}


void
TService::SetStatus(uint32 status, uint32 error)
{
  // notify the system of our current status
  //
  StatusCode = status;
  ErrorCode = error;
  SERVICE_STATUS ss;
  ss.dwServiceType             = SERVICE_WIN32_OWN_PROCESS;
  ss.dwCurrentState            = StatusCode;
  ss.dwControlsAccepted        = SERVICE_ACCEPT_STOP |
                                 SERVICE_ACCEPT_PAUSE_CONTINUE;
  ss.dwWin32ExitCode           = (ErrorCode ? ERROR_SERVICE_SPECIFIC_ERROR :
                                              NO_ERROR);
  ss.dwServiceSpecificExitCode = ErrorCode;
  ss.dwCheckPoint              = 0;
  ss.dwWaitHint                = 1000;
  SetServiceStatus(HStatus, &ss);
}


bool
TService::CheckStop()
{
  TEventSemaphore::TLock checkLock(PauseEvent, 0);
  if (!checkLock.WasAquired()) {
    // the user has requested that the service be paused
    //
    PauseBegin();
    SetStatus(SERVICE_PAUSED);
    TEventSemaphore::TLock waitLock(PauseEvent);
    PauseEnd();
    SetStatus(SERVICE_RUNNING);
  }
  
  return StopFlag;
}


void
TService::SetStop(bool stop)
{
  StopFlag = stop;
}


void
TService::Log(const char* str, TLogType type)
{
  // register an event source if we have not already done so
  //
  if (!HEventLog)
    HEventLog = RegisterEventSource(NULL, GetName());

  // write the message to the event log
  //
  const char* msgs[1];
  msgs[0] = str;
  ReportEvent(HEventLog, (WORD)type, 0, 0, NULL, 1, 0, msgs, NULL);
}


void
TService::ServiceMain(uint32 argc, char* argv[])
{
  HStatus = RegisterServiceCtrlHandler(GetName(), ::ControlHandler);
  if (!HStatus) {
    SetStatus(SERVICE_STOPPED, -1);
    Log("Unable to register control handler.");
    return;
  }

  // reset all flags and things for a fresh start
  //
  SetStatus(SERVICE_START_PENDING);
  PauseEvent.Set();
  StopFlag = false;
  ArgC = argc;
  ArgV = argv;

  // intialise and run the service
  //
  if (Init()) {
    SetStatus(SERVICE_RUNNING);
    Run();
    SetStatus(SERVICE_STOP_PENDING);
    Cleanup();
    SetStatus(SERVICE_STOPPED);
  }
  else
    SetStatus(SERVICE_STOPPED, -1);
}


void
TService::ControlHandler(DWORD control)
{
  switch (control)
  {
    case SERVICE_CONTROL_STOP:
      SetStatus(SERVICE_STOP_PENDING);
      StopFlag = true;
      PauseEvent.Set();
      break;

    case SERVICE_CONTROL_PAUSE:
      SetStatus(SERVICE_PAUSE_PENDING);
      PauseEvent.Reset();
      break;

    case SERVICE_CONTROL_CONTINUE:
      SetStatus(SERVICE_CONTINUE_PENDING);
      PauseEvent.Set();
      break;

    case SERVICE_CONTROL_INTERROGATE:
      SetStatus(StatusCode, ErrorCode);
      break;

    case SERVICE_CONTROL_SHUTDOWN:
      SetStatus(SERVICE_STOP_PENDING);
      StopFlag = true;
      PauseEvent.Set();
      break;
  }
}
